/* 
 *  InteractionDataGUI.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.avaya.ccs.api.CodeListI;
import com.avaya.ccs.api.ContactNumberI;
import com.avaya.ccs.api.CustomFieldListI;
import com.avaya.ccs.api.enums.ContactType;
import com.avaya.ccs.api.enums.InteractionState;
import com.avaya.ccs.api.enums.ResourceState;

import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonBar;
import javafx.scene.control.ButtonType;
import javafx.scene.control.ComboBox;
import javafx.scene.control.Dialog;
import javafx.scene.control.Label;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.control.MenuItem;
import javafx.scene.control.TableColumn;
import javafx.scene.control.cell.PropertyValueFactory;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import javafx.util.Callback;

public class InteractionDataGUI extends TableDataGUI {

    private static final String HOLD = "Hold";
    private static final String UNHOLD = "UnHold";
    private static final String INIT_TRANSFER = "Transfer";
    private static final String COMPLETE_TRANSFER = "Complete Transfer";
    private static final String INIT_CONFERENCE = "Conf";
    private static final String COMPLETE_CONFERENCE = "Complete Conf";
    private static final String WRAPUP = "Wrapup";

    private HBox intreractionButtonHolderTop;
    private Button buttonAnswer;
    private Button buttonReject;
    
    private Button buttonHoldUnHold;
    private Button buttonEnd;
    private Button buttonDTMF;
    private Button buttonActivity;

    private HBox intreractionButtonHolderBottom;
    private Button buttonConference;
    private Button buttonJoin;
    private Button buttonTransfer;
    private Button buttonEmergency;

    private HBox pomButtonHolderTop;
    private Button buttonWrapup;
    private Button buttonWrapupWithCode;
    private Button buttonExtendWrapup;
    private Button buttonPreviewDial;
    private Button buttonReDial;

    private HBox pomButtonHolderBottom;
    private Button buttonAddToDNC;
    private Button buttonCallback;
    private Button buttonChangeConfOwner;
    private Button buttonEndConference;

    private VBox interactionTable;

    private final SideBar sideBar;

    private static final Logger LOG = Logger.getLogger(InteractionDataGUI.class);

    private final InteractionExecutor interactionExe;
    private final CodeSelectionForm codeSelection;
    private final PreviewDialForm previewDial;
    private final ConsultForm consultForm;
    
    private final Map<String,EmailForm> emailForms;
    private final Map<String,WebchatForm> webchatForms;
    
    private final JoinForm joinForm;
    private final DtmfKeypad dtmfKeypad;
    private final CallbackForm callbackForm;
	private String css;
    private UserDataGUI userGui;
    private boolean isCtiWasLost;

    public InteractionDataGUI(UserDataGUI userGui, InteractionExecutor interactionExe, String css, CodeSelectionForm codeSelection,
          PreviewDialForm previewDial, ConsultForm consultForm, JoinForm joinForm, CallbackForm callbackForm) {
        super(LOG);
        this.userGui=userGui;
        this.interactionExe = interactionExe;
        this.css=css;
        this.codeSelection = codeSelection;
        this.previewDial = previewDial;
        this.consultForm = consultForm;
        this.joinForm = joinForm;
        this.callbackForm = callbackForm;
        this.isCtiWasLost = false;

        emailForms= new HashMap<String, EmailForm>();
        webchatForms= new HashMap<String, WebchatForm>();
        setDetailsForm(new InteractionDetailsForm(interactionExe, css, this));
        dtmfKeypad = new DtmfKeypad(interactionExe);

        createInteractionButtons();
        createPomButtons();
        createInteractionTable();
        sideBar = new SideBar(interactionExe, this);

        redrawControls();
    }

    public VBox getInteractionButtons() {
        return new VBox(intreractionButtonHolderTop, intreractionButtonHolderBottom);
    }

    public VBox getPOMButtons() {
        return new VBox(pomButtonHolderTop, pomButtonHolderBottom);
    }

    public VBox getInteractionTable() {
        return interactionTable;
    }

    private void createInteractionButtons() {
        intreractionButtonHolderTop = new HBox();
        intreractionButtonHolderTop.getStyleClass().add("hbox");

        buttonAnswer = new Button("Answer");
        buttonAnswer.getStyleClass().add("interactionButton");
        
        buttonAnswer.setOnAction((ActionEvent e) -> {
            String methodName = "answerButton() ";
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info(methodName + "ID:" + interactionID);
            interactionExe.answer(interactionID);
             	
            ContactType type = getCurrentInteractionSelection().getContactType();
            if (type==ContactType.EMail) {
            	EmailForm newForm= new EmailForm(interactionExe, css, getCurrentInteractionSelection().getUserID(), false);
            	newForm.setAgentSignature(userGui.getUser(getCurrentInteractionSelection().getUserID()).getAgentSignature());
            	emailForms.put(interactionID, newForm);
            	emailForms.get(interactionID).setInteractionData(getCurrentInteractionSelection());
            } else if (type==ContactType.Web_Communications) {
            	WebchatForm newWebchatForm= new WebchatForm(interactionID,interactionExe, interactionID);
            	webchatForms.put(interactionID, newWebchatForm);
            	webchatForms.get(interactionID).setInteractionData(getCurrentInteractionSelection());
            }
        });
        
        buttonReject=new Button("Reject");
        buttonReject.getStyleClass().add("interactionButton");
        buttonReject.setOnAction((ActionEvent e) -> {
            String methodName = "rejectButton() ";
            LOG.info(methodName);
            userGui.setAgentNotReady();
        });
        

        buttonHoldUnHold = new Button(HOLD);
        buttonHoldUnHold.getStyleClass().add("interactionButton");
        buttonHoldUnHold.setOnAction((ActionEvent e) -> {
            String methodName = "holdButton() ";
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info(methodName + "ID:" + interactionID);
            if (getCurrentInteractionSelection().canHold()) {
                interactionExe.hold(interactionID);
                LOG.info(methodName + "Hold:ID:" + interactionID);
            } else if (getCurrentInteractionSelection().canUnhold()) {
                LOG.info(methodName + "UnHold:ID:" + interactionID);
                interactionExe.unHold(interactionID);
            } else {
                LOG.error(methodName + "Hold/UnHold button pressed but both capabilities false");
            }
        });

        buttonEnd = new Button("End");
        buttonEnd.getStyleClass().add("interactionButton");
        buttonEnd.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("EndButton() ID:" + interactionID);
            interactionExe.end(interactionID);
        });

        buttonDTMF = new Button("DTMF");
        buttonDTMF.getStyleClass().add("interactionButton");
        buttonDTMF.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("DTMFButton() ID:" + interactionID);
            dtmfKeypad.show(interactionID);
        });

        buttonActivity = new Button("Activity");
        buttonActivity.getStyleClass().add("interactionButton");
        buttonActivity.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("ActivityButton() ID:" + interactionID);
            interactionExe.getActivityCodes(interactionID);
            // Display code selection form, this will initial be blank
            codeSelection.show();
        });

        intreractionButtonHolderBottom = new HBox();
        intreractionButtonHolderBottom.getStyleClass().add("hbox");

        buttonConference = new Button(INIT_CONFERENCE);
        buttonConference.getStyleClass().add("interactionButtonWide");
        buttonConference.setOnAction((ActionEvent e) -> {
            String methodName = "conferenceButton() ";
            InteractionData interaction = getCurrentInteractionSelection();
            String interactionID = interaction.getId();
            LOG.info(methodName + " ID:" + interactionID);
            
            if (interaction.canInitiateConference()) {
                getConsultForm().show();
                getConsultForm().initializeConsultForm(getCurrentInteractionSelection());

                if (interaction.canInitiateTransferToDestination()) {
                    // If the capability to transfer to a destination is set
                    // then the consult destination list can be used
                    interactionExe.getConsultDestinations(interactionID);
                }
            } else if (interaction.canCompleteConference()) {
                // complete conference
                interactionExe.completeConference(interactionID);
            } else {
                LOG.error(methodName + "no init or complete conference capabilities");
            }
        });

        buttonJoin = new Button("Join");
        buttonJoin.getStyleClass().add("interactionButton");
        buttonJoin.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("JoinButton() ID:" + interactionID);
            joinForm.updateJoinForm(getUnselectedInteractions(), interactionID);
            joinForm.show();
        });

        buttonTransfer = new Button(INIT_TRANSFER);
        buttonTransfer.getStyleClass().add("interactionButtonWide");
        buttonTransfer.setOnAction((ActionEvent e) -> {
            String methodName = "transferButton() ";
            InteractionData interaction = getCurrentInteractionSelection();
            String interactionID = interaction.getId();
            ContactType contactType = interaction.getContactType();

            if (interaction.canInitiateTransfer()) {
                LOG.info(methodName + " Init:ID:" + interactionID);
                getConsultForm().show();
                getConsultForm().initializeTransferForm(getCurrentInteractionSelection());
                getConsultForm().setCurrentInteractionId(interactionID);
                
                // aryabin - 22/08/2019 - CC-18626
                // canInitiateTransferToDestination capability exist only for POM contacts
                if (contactType == ContactType.POM_Outbound || interaction.canInitiateTransferToDestination()) {
					// If the capability to transfer to a destination is set
					// then the consult destination list can be used
                    interactionExe.getConsultDestinations(interactionID);
                }
            } else if (interaction.canCompleteTransfer()) {
                LOG.info(methodName + " Complete:ID:" + interactionID);
                // complete transfer
                interactionExe.completeTransfer(interactionID);
            } else {
                LOG.error(methodName + " no init or complete transfer capabilities");
            }
        });

        buttonEmergency = new Button("Emergency");
        buttonEmergency.getStyleClass().add("interactionButtonWide");
        buttonEmergency.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("EmergencyButton() ID:" + interactionID);
            interactionExe.emergency(interactionID);
        });

        intreractionButtonHolderTop.getChildren().addAll(buttonAnswer, buttonReject, buttonHoldUnHold, buttonEnd, buttonDTMF,
                buttonActivity);
        intreractionButtonHolderBottom.getChildren().addAll(buttonConference, buttonJoin, buttonTransfer,
                buttonEmergency);
        
        
        
    }

    private void createPomButtons() {
        pomButtonHolderTop = new HBox();
        pomButtonHolderTop.getStyleClass().add("hbox");

        pomButtonHolderBottom = new HBox();
        pomButtonHolderBottom.getStyleClass().add("hbox");

        buttonWrapup = new Button(WRAPUP);
        buttonWrapup.getStyleClass().add("interactionButton");
        buttonWrapup.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("WrapUpButton() ID:" + interactionID);
            interactionExe.wrapUP(interactionID);
        });

        buttonWrapupWithCode = new Button("WrapupCode");
        buttonWrapupWithCode.getStyleClass().add("interactionButtonWide");
        buttonWrapupWithCode.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("WrapUpWithCodeButton() ID:" + interactionID);
            interactionExe.getDispositionCodes(interactionID);
            // Display code selection form, this will initiall be blank
            codeSelection.show();
        });

        buttonExtendWrapup = new Button("Extend Wrapup");
        buttonExtendWrapup.getStyleClass().add("interactionButtonWide");
        buttonExtendWrapup.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("ExtendWrapUpButton() ID:" + interactionID);
            interactionExe.extendWrapup(interactionID);
        });

        buttonPreviewDial = new Button("Preview");
        buttonPreviewDial.getStyleClass().add("interactionButton");
        buttonPreviewDial.setOnAction((ActionEvent e) -> {
            LOG.info("PreviewButton()");
            previewDial.show();
            previewDial.updatePreviewDialForm(getCurrentInteractionSelection(), getCallbackForm());
        });

        buttonReDial = new Button("Redial");
        buttonReDial.getStyleClass().add("interactionButton");
        buttonReDial.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("RedialButton() ID:" + interactionID);
            List<ContactNumberI> numbers = getCurrentInteractionSelection().getContactNumbers();
            redialDialog(interactionID, numbers);
        });

        buttonAddToDNC = new Button("DNC");
        buttonAddToDNC.getStyleClass().add("interactionButton");
        buttonAddToDNC.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("DNCButton() ID:" + interactionID);
            interactionExe.addToDNC(interactionID);
        });

        buttonCallback = new Button("Callback");
        buttonCallback.getStyleClass().add("interactionButton");
        buttonCallback.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("CallbackButton() ID:" + interactionID);
            getCallbackForm().initializeForm(getCurrentInteractionSelection());
            interactionExe.getCallbackDestinations(interactionID);
            getCallbackForm().show();
        });

        buttonChangeConfOwner = new Button("Change Conf");
        buttonChangeConfOwner.getStyleClass().add("interactionButtonWide");
        buttonChangeConfOwner.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("ChangeConfButton() ID:" + interactionID);
            interactionExe.changeConferenceOwner(interactionID);
        });

        buttonEndConference = new Button("End Conf");
        buttonEndConference.getStyleClass().add("interactionButton");
        buttonEndConference.setOnAction((ActionEvent e) -> {
            String interactionID = getCurrentInteractionSelection().getId();
            LOG.info("EndConfButton() ID:" + interactionID);
            interactionExe.endConference(interactionID);
        });
        pomButtonHolderTop.getChildren().addAll(buttonWrapup, buttonWrapupWithCode, buttonExtendWrapup,
                buttonPreviewDial, buttonReDial);
        pomButtonHolderBottom.getChildren().addAll(buttonAddToDNC, buttonCallback, buttonChangeConfOwner,
                buttonEndConference);
    }

    private void createInteractionTable() {
        getItemSelection().setEditable(false);
        getItemSelection().setPlaceholder(new Label("No interactions"));
        TableColumn<Data, String> address = new TableColumn<Data, String>("Address");
        address.getStyleClass().add("interactionTableRows");
        address.setCellValueFactory(new PropertyValueFactory<>("addressName"));

        TableColumn<Data, String> contactID = new TableColumn<Data, String>("ContactID");
        contactID.getStyleClass().add("interactionTableRows");
        contactID.setCellValueFactory(new PropertyValueFactory<>("contactID"));
        TableColumn<Data, String> calledAddressName = new TableColumn<Data, String>("Called");
        calledAddressName.getStyleClass().add("interactionTableRows");
        calledAddressName.setCellValueFactory(new PropertyValueFactory<>("calledAddressName"));

        TableColumn<Data, String> callingAddressName = new TableColumn<Data, String>("Calling");
        callingAddressName.getStyleClass().add("interactionTableRows");
        callingAddressName.setCellValueFactory(new PropertyValueFactory<>("callingAddressName"));

        TableColumn<Data, String> state = new TableColumn<Data, String>("State");
        state.getStyleClass().add("interactionTableRows");
        state.setCellValueFactory(new PropertyValueFactory<>("combinedState"));

        TableColumn<Data, String> stateReason = new TableColumn<Data, String>("Reason");
        stateReason.getStyleClass().add("interactionTableRowsReason");
        stateReason.setCellValueFactory(new PropertyValueFactory<>("stateReason"));

        TableColumn<Data, String> activityCode = new TableColumn<Data, String>("Activity");
        activityCode.getStyleClass().add("interactionTableRows");
        activityCode.setCellValueFactory(new PropertyValueFactory<>("activityCodeNumber"));

        getItemSelection().setItems(getDetailsForm().getItems());

        getItemSelection().getColumns().add(address);
        getItemSelection().getColumns().add(contactID);
        getItemSelection().getColumns().add(state);
        getItemSelection().getColumns().add(stateReason);
        getItemSelection().getColumns().add(calledAddressName);
        getItemSelection().getColumns().add(callingAddressName);
        getItemSelection().getColumns().add(activityCode);
        getItemSelection().setPrefHeight(5000.0);

        interactionTable = new VBox();
        interactionTable.getStyleClass().add("interactionTable");
        interactionTable.getChildren().addAll(getItemSelection());
    }

    @Override
    protected void redrawControls() {
    	LOG.debug("redrawControls");
        InteractionData data = getCurrentInteractionSelection();
        buttonAnswer.setDisable(!data.canAnswer());
        buttonReject.setDisable(!data.canAnswer());
        buttonEnd.setDisable(!data.canEnd()); 
        
        buttonEmergency.setDisable(!data.canEmergency());
        // While hold and unhold are separate capabilities it should only be
        // possible
        // for one of them to be true at the same time
        if (data.canHold()) {
            buttonHoldUnHold.setText(HOLD);
            buttonHoldUnHold.setDisable(false);
        } else if (data.canUnhold()) {
            buttonHoldUnHold.setDisable(false);
            buttonHoldUnHold.setText(UNHOLD);
        } else {
            // if both capabilities are false disable button with HOLD text
            buttonHoldUnHold.setText(HOLD);
            buttonHoldUnHold.setDisable(true);
        }

        // The transfer button is used to initiate or complete a transfer
        if (data.canInitiateTransfer()) {
            buttonTransfer.setDisable(false);
            buttonTransfer.setText(INIT_TRANSFER);
        } else if (data.canCompleteTransfer()) {
            buttonTransfer.setDisable(false);
            buttonTransfer.setText(COMPLETE_TRANSFER);
        } else {
            buttonTransfer.setDisable(true);
            buttonTransfer.setText(INIT_TRANSFER);
        }

        // The conference button is used to initiate or complete a conference
        if (data.canInitiateConference()) {
            buttonConference.setDisable(false);
            buttonConference.setText(INIT_CONFERENCE);
        } else if (data.canCompleteConference()) {
            buttonConference.setDisable(false);
            buttonConference.setText(COMPLETE_CONFERENCE);
        } else {
            buttonConference.setDisable(true);
            buttonConference.setText(INIT_CONFERENCE);
        }
        buttonJoin.setDisable(!data.canJoinConference());
        buttonActivity.setDisable(!data.canSetActivityCode());
        buttonDTMF.setDisable(!data.canPlayDTMF());

        // Enable POM buttons based on capabilities
        buttonWrapup.setDisable(!data.canWrapupCall());
        buttonWrapupWithCode.setDisable(!data.canWrapupCallWithCode());
        buttonExtendWrapup.setDisable(!data.canExtendWrapup());
        buttonWrapup.setText(WRAPUP);
        buttonCallback.setDisable(!data.canCreateCallback());
        buttonPreviewDial.setDisable(!data.canPreviewDial());
        buttonReDial.setDisable(!data.canRedial());
        buttonAddToDNC.setDisable(!data.canAddToDnc());
        buttonChangeConfOwner.setDisable(!data.canChangeConferenceOwner());
        buttonEndConference.setDisable(!data.canEndConference());
        
        this.sideBar.update(data);
    }

    public void handleAddressStateEvent(ResourceData data) {
        String methodName = "handleAddressStateEvent() ";
        LOG.debug(methodName + "+");
        LOG.debug(methodName + "Address:" + data.getAddress() + " State:" + data.getState().toString());
        ResourceState state = data.getState();
        if (state == ResourceState.OutOfService) {
            isCtiWasLost = true;
            LOG.errorStatus("Address " + data.getAddress() + " is Out Of Service. Reason: " + data.getStateReason() + ". Message: " + data.getStateReasonMessage());
        } else if ((state == ResourceState.InService) && isCtiWasLost == true) {
            isCtiWasLost = false;
            LOG.infoStatus("Address " + data.getAddress() + " is In Service. Reason: " + data.getStateReason());
        }
        LOG.debug(methodName + "-");
    }

    public MenuItem getMenuItem() {
        return getDetailsForm().getMenuItem();
    }

    // Return the current user selection or a dummy value if there is none....
    protected InteractionData getCurrentInteractionSelection() {
        String methodName = "getCurrentInteractionSelection() ";
        Data currentInteraction = getCurrentItemSelection();

        if (currentInteraction != null) {
            return (InteractionData) currentInteraction;
        } else {
            LOG.trace(methodName + "current selection is NULL, returning blank");
            return new InteractionData(null);
        }
    }

    private List<InteractionData> getUnselectedInteractions() {
        List<Data> unselectedData = getUnselectedItems();
        List<InteractionData> unselectedInteractions = new ArrayList<>();
        for (Data d : unselectedData) {
            unselectedInteractions.add((InteractionData) d);
        }
        return unselectedInteractions;
    }

    public void UpdateInteractionData(List<InteractionData> data) {
        // Need to convert to use data objects before setting
        ObservableList<Data> newData = FXCollections.observableArrayList();
        newData.addAll(data);
        this.update(newData);
    }

    public void UpdateInteractionData(InteractionData data) {
        // Need to convert to use data objects before setting
    	Data newData = data;
        this.update(newData);
    }

    public void updateCodeSectionForm(CodeListI codes) {
        codeSelection.updateCodeSectionForm(codes, getCurrentInteractionSelection().getId());
    }

    public void updateCustomFields(CustomFieldListI customFieldList) {
        this.getSideBar().updateCustomFields(customFieldList);
    }

    public SideBar getSideBar() {
    	return sideBar;
    }

    
    public VBox getSideBarVbox() {
    	return this.sideBar.getSideBar();
    }

    public ConsultForm getConsultForm() {
        return consultForm;
    }
    
    public EmailForm getEmailFormForContact(String id){
    	boolean isAdHocEmail = getCurrentInteractionSelection().getState() == InteractionState.Initiated;
    	//if it doesn't exist, create one
    	if (emailForms.get(id)==null) {
    		emailForms.put(id, new EmailForm(interactionExe, getCurrentInteractionSelection().getUserID(), css, isAdHocEmail));
    		emailForms.get(id).setInteractionData(getCurrentInteractionSelection());
    	}
        return emailForms.get(id);
    }
    
    public void createAndAddNewWebchatForm(String id) {
    	Platform.runLater(() -> {
    		LOG.info("createAndAddNewWebchatForm() " +id );
        	WebchatForm newform= new WebchatForm(id, interactionExe, css);
    		webchatForms.put(id, newform);
    	});

    }
    
    public void removeAndCloseWebChatForm(String id) {
		Platform.runLater(() -> {
        	LOG.info("removeAndCloseWebChatForm() " +id );
        	WebchatForm thisform= webchatForms.get(id);
        	thisform.close();
    		webchatForms.remove(id);
    	});
    }
    
    public WebchatForm getWebChatFormForContact(String id) {
    	return webchatForms.get(id);
	}
    

    public CallbackForm getCallbackForm() {
        return callbackForm;
    }
    
    public InteractionData getSelectedInteractionData() {
    	return this.getSelectedInteractionData();
    }
    
    public void deleteEmailFormCorContact(String id) {
		LOG.info("deleteEmailFormCorContact"," removing emailform from map for contact: "+id);
		emailForms.remove(id);	
		getCurrentInteractionSelection().setDeleted(true);
	}
    
    public void customerDisconnectedWebChat(String contactId) {
		LOG.debug("customerDisconnectedWebChat" + "  contact: "+contactId);
		Data item = getItem(contactId);
	
		WebchatForm form = webchatForms.get(contactId);
		Platform.runLater(()->{
			form.goToWrapup();
		});
		item.setDeleted(true);
	}
    
    private void redialDialog(String interactionID, List<ContactNumberI> numbers) {
        String methodName = "redialDialog() ";
        Dialog<ContactNumberI> dialog = new Dialog<>();
        dialog.setTitle("Redial Dialog");

        ButtonType redialButtonType = new ButtonType("Redial", ButtonBar.ButtonData.OK_DONE);
        dialog.getDialogPane().getButtonTypes().addAll(redialButtonType, ButtonType.CANCEL);

        GridPane grid = new GridPane();
        grid.getStyleClass().add("dialog");

        ComboBox<ContactNumberI> numberList = new ComboBox<ContactNumberI>(FXCollections.observableList(numbers));
        numberList.getSelectionModel().selectFirst();

        numberList.setCellFactory(new Callback<ListView<ContactNumberI>, ListCell<ContactNumberI>>() {
            @Override
            public ListCell<ContactNumberI> call(ListView<ContactNumberI> list) {
                return new InteractionDataGUI.CodeCell();
            }
        });
        numberList.setButtonCell(new InteractionDataGUI.CodeCell());

        grid.add(new Label("POM Zone:"), 0, 2);
        grid.add(numberList, 1, 2);

        dialog.getDialogPane().setContent(grid);

        dialog.setResultConverter(dialogButton -> {
            if (dialogButton == redialButtonType) {
                return numberList.getValue();
            }
            return null;
        });

        Optional<ContactNumberI> result = dialog.showAndWait();

        result.ifPresent(number -> {
            LOG.info(methodName + "Name:" + number.getName() + " Number:" + number.getNumber());
            interactionExe.redial(interactionID, number);
        });
    }

    private static class CodeCell extends ListCell<ContactNumberI> {

        @Override
        public void updateItem(ContactNumberI item, boolean empty) {
            super.updateItem(item, empty);
            if (item != null) {
                setText(item.getName() + ":" + item.getNumber());
            }
        }
    }

	
}
